/** @file
  Extracts system info from EC, for Mobile systems.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2016 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/


#include <SetupPrivate.h>
#include <Library/EcMiscLib.h>

/**

  @param[in]  Reg
  @param[out] RetValue

  @retval    EFI_STATUS
**/
EFI_STATUS
ReadEcRegister(
    IN  UINT8  Reg,
    OUT UINT8  *RetValue
)
{
  EFI_STATUS Status;
  UINT8      DataBuffer[1];

  DataBuffer[0] = Reg;
  Status = ReadEcRam (DataBuffer);
  if (Status == EFI_SUCCESS) {
    *RetValue = DataBuffer[0];
  }
  return Status;
}


/**
  Returns system monitor values from the EC.

  @param[in]  KeyValue
  @param[out] MonitorValues  Pointer to a structure to return the monitor values in.

  @retval     EFI_INVALID_PARAMETER
  @retval     EFI_SUCCESS
**/
EFI_STATUS
GetSystemMonitorValues (
    IN  UINT16                             KeyValue,
    OUT MOBILE_SYSTEM_MONITOR_INFO*        MonitorValues
  )
{
  // Local Variables
  UINT8                     Data;
  UINT8                     Register;
  EFI_STATUS                Status;
  PLATFORM_INFO             *PlatformInfo;

  // Check input parameters
  if (MonitorValues == NULL) {
    return EFI_INVALID_PARAMETER;
  }
  Status = gBS->LocateProtocol (
                &gPlatformInfoProtocolGuid,
                NULL,
                (VOID **) &PlatformInfo
                );
  if (!EFI_ERROR (Status)) {
    if (!PlatformInfo->EcPresent) {
      return EFI_SUCCESS;
    }
  } else {
    DEBUG ((DEBUG_ERROR, "Failed to locate PlatformInfo Protocol!\n"));
    ASSERT_EFI_ERROR (Status);
    return Status;
  }
  switch(KeyValue)
  {
    case THERMAL_SENSOR_1_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR1;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR1 = Data;
          break;
        }
    case THERMAL_SENSOR_2_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR2;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR2 = Data;
          break;
        }
    case THERMAL_SENSOR_3_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR3;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR3 = Data;
          break;
        }
    case THERMAL_SENSOR_4_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR4;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR4 = Data;
          break;
        }
    case THERMAL_SENSOR_5_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR5;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR5 = Data;
          break;
        }
    case THERMAL_SENSOR_6_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR6;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR6 = Data;
          break;
        }
    case THERMAL_SENSOR_7_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR7;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR7 = Data;
          break;
        }
    case THERMAL_SENSOR_8_KEY:
    {
      //
      // Read Thermal Sensor from EC
      //
      Register = EC_REG_TSR8;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TSR8 = Data;
          break;
        }

    case THERMAL_THERMISTOR_1_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER1;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER1 = Data << 2;
          break;
        }
    case THERMAL_THERMISTOR_2_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER2;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER2 = Data << 2;
          break;
        }
    case THERMAL_THERMISTOR_3_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER3;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER3 = Data << 2;
          break;
        }
    case THERMAL_THERMISTOR_4_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER4;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER4 = Data << 2;
          break;
        }
    case THERMAL_THERMISTOR_5_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER5;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER5 = Data << 2;
          break;
        }
    case THERMAL_THERMISTOR_6_KEY:
    {
      //
      // Read Thermistor from EC
      //
      Register = EC_REG_TER6;
      Data = 0;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->EC_TER6 = Data << 2;
          break;
        }

    case CPU_FAN_KEY :
    {
      //
      // Read CPU Fan Speed from EC
      //
      Register = EC_REG_CPU_FAN_SPEED + 1;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->CPUFanSpeed = Data;
      Register = EC_REG_CPU_FAN_SPEED;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->CPUFanSpeed = (MonitorValues->CPUFanSpeed << 8) + Data;
          break;
        }
    case PCH_DTS_TEMP_KEY :
    {
      //
      // Read PCH DTS Temp from EC
      //
      Register = EC_REG_PCH_DTS_TEMP;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->PCHDTSTemp = Data;
          break;
        }
    case TS_ON_DIMM0_TEMP_KEY :
    {
      //
      // Read TS-on-DIMM0 Temp from EC
      //
      Register = EC_REG_TS_ON_DIMM0_TEMP;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->TSonDimm0Temp = Data;
          break;
        }
    case TS_ON_DIMM1_TEMP_KEY :
    {
      //
      // Read TS-on-DIMM1 Temp from EC
      //
      Register = EC_REG_TS_ON_DIMM1_TEMP;
      Status = ReadEcRegister (Register, &Data);
      MonitorValues->TSonDimm1Temp = Data;
          break;
        }
        default:
          break;
  }

  return EFI_SUCCESS;
}


/**

  @param[in] HiiHandle
  @param[in] Class
**/
VOID
InitHhmMobileStrings (
  IN EFI_HII_HANDLE HiiHandle,
  IN UINT16         Class
  )
{
    MOBILE_SYSTEM_MONITOR_INFO         MonitorValues;

  if (Class != ADVANCED_FORM_SET_CLASS) {
    return ;
  }

  GetSystemMonitorValues (THERMAL_SENSOR_1_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_2_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_3_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_4_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_5_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_6_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_7_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_SENSOR_8_KEY, &MonitorValues);

  GetSystemMonitorValues (THERMAL_THERMISTOR_1_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_THERMISTOR_2_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_THERMISTOR_3_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_THERMISTOR_4_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_THERMISTOR_5_KEY, &MonitorValues);
  GetSystemMonitorValues (THERMAL_THERMISTOR_6_KEY, &MonitorValues);

  GetSystemMonitorValues (CPU_FAN_KEY, &MonitorValues);
  GetSystemMonitorValues (PCH_DTS_TEMP_KEY, &MonitorValues);
  GetSystemMonitorValues (TS_ON_DIMM0_TEMP_KEY, &MonitorValues);
  GetSystemMonitorValues (TS_ON_DIMM1_TEMP_KEY, &MonitorValues);

  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_1_VALUE), L"%d C", MonitorValues.EC_TSR1);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_2_VALUE), L"%d C", MonitorValues.EC_TSR2);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_3_VALUE), L"%d C", MonitorValues.EC_TSR3);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_4_VALUE), L"%d C", MonitorValues.EC_TSR4);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_5_VALUE), L"%d C", MonitorValues.EC_TSR5);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_6_VALUE), L"%d C", MonitorValues.EC_TSR6);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_7_VALUE), L"%d C", MonitorValues.EC_TSR7);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_8_VALUE), L"%d C", MonitorValues.EC_TSR8);

  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_1_VALUE), L"%d raw", MonitorValues.EC_TER1);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_2_VALUE), L"%d raw", MonitorValues.EC_TER2);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_3_VALUE), L"%d raw", MonitorValues.EC_TER3);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_4_VALUE), L"%d raw", MonitorValues.EC_TER4);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_5_VALUE), L"%d raw", MonitorValues.EC_TER5);
  InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_6_VALUE), L"%d raw", MonitorValues.EC_TER6);

  InitString(HiiHandle, STRING_TOKEN(STR_CPU_FAN_VALUE), L"%d rpm", MonitorValues.CPUFanSpeed);
  InitString(HiiHandle, STRING_TOKEN(STR_PCH_DTS_TEMP_VALUE), L"%d C", MonitorValues.PCHDTSTemp);

}


/**

  @param[in] HiiHandle
  @param[in] Class
  @param[in] SubClass
  @param[in] Key
**/
VOID HhmMobileCallBack(
  IN EFI_HII_HANDLE HiiHandle,
  IN UINT16 Class,
  IN UINT16 SubClass,
  IN UINT16 Key
  )
{
    MOBILE_SYSTEM_MONITOR_INFO         MonitorValues;

    GetSystemMonitorValues (Key, &MonitorValues);

    switch(Key)
    {
        case THERMAL_SENSOR_1_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_1_VALUE), L"%d C", MonitorValues.EC_TSR1);
            break;
        }
        case THERMAL_SENSOR_2_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_2_VALUE), L"%d C", MonitorValues.EC_TSR2);
            break;
        }
        case THERMAL_SENSOR_3_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_3_VALUE), L"%d C", MonitorValues.EC_TSR3);
            break;
        }
        case THERMAL_SENSOR_4_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_4_VALUE), L"%d C", MonitorValues.EC_TSR4);
            break;
        }
        case THERMAL_SENSOR_5_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_5_VALUE), L"%d C", MonitorValues.EC_TSR5);
            break;
        }
        case THERMAL_SENSOR_6_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_6_VALUE), L"%d C", MonitorValues.EC_TSR6);
            break;
        }
        case THERMAL_SENSOR_7_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_7_VALUE), L"%d C", MonitorValues.EC_TSR7);
            break;
        }
        case THERMAL_SENSOR_8_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_SENSOR_8_VALUE), L"%d C", MonitorValues.EC_TSR8);
            break;
        }

        case THERMAL_THERMISTOR_1_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_1_VALUE), L"%d raw", MonitorValues.EC_TER1);
            break;
        }
        case THERMAL_THERMISTOR_2_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_2_VALUE), L"%d raw", MonitorValues.EC_TER2);
            break;
        }
        case THERMAL_THERMISTOR_3_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_3_VALUE), L"%d raw", MonitorValues.EC_TER3);
            break;
        }
        case THERMAL_THERMISTOR_4_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_4_VALUE), L"%d raw", MonitorValues.EC_TER4);
            break;
        }
        case THERMAL_THERMISTOR_5_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_5_VALUE), L"%d raw", MonitorValues.EC_TER5);
            break;
        }
        case THERMAL_THERMISTOR_6_KEY:
        {
            InitString(HiiHandle, STRING_TOKEN(STR_THERMAL_THERMISTOR_6_VALUE), L"%d raw", MonitorValues.EC_TER6);
            break;
        }

        case CPU_FAN_KEY :
        {
            InitString(HiiHandle, STRING_TOKEN(STR_CPU_FAN_VALUE), L"%d rpm", MonitorValues.CPUFanSpeed);
            break;
        }
        case PCH_DTS_TEMP_KEY :
        {
            InitString(HiiHandle, STRING_TOKEN(STR_PCH_DTS_TEMP_VALUE), L"%d C", MonitorValues.PCHDTSTemp);
            break;
        }
        case TS_ON_DIMM0_TEMP_KEY :
        {
            InitString(HiiHandle, STRING_TOKEN(STR_TS_ON_DIMM0_TEMP_VALUE), L"%d C", MonitorValues.TSonDimm0Temp);
            break;
        }
        case TS_ON_DIMM1_TEMP_KEY :
        {
            InitString(HiiHandle, STRING_TOKEN(STR_TS_ON_DIMM1_TEMP_VALUE), L"%d C", MonitorValues.TSonDimm1Temp);
            break;
        }
  }
}
